<?php

namespace App\Http\Controllers;

use App\Models\Doctor;
use App\Models\Patient;
use App\Models\Prescription;
use App\Models\Appointment;
use App\Models\Branch;
use App\Models\User;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportsController extends Controller
{
    public function index()
    {
        return view('admin.reports.index');
    }

    public function prescriptions(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->toDateString());
        $endDate = $request->input('end_date', Carbon::now()->toDateString());

        $prescriptions = Prescription::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->with(['doctor', 'patient', 'appointment'])
            ->get();

        // Statistics
        $stats = [
            'total' => $prescriptions->count(),
            'this_month' => Prescription::whereMonth('created_at', Carbon::now()->month)->count(),
            'last_month' => Prescription::whereMonth('created_at', Carbon::now()->subMonth()->month)->count(),
            'this_year' => Prescription::whereYear('created_at', Carbon::now()->year)->count(),
        ];

        // Daily prescription count for chart
        $dailyData = Prescription::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Prescriptions by doctor
        $byDoctor = Prescription::whereBetween('prescriptions.created_at', [$startDate, $endDate . ' 23:59:59'])
            ->join('doctors', 'prescriptions.doctor_id', '=', 'doctors.id')
            ->selectRaw('CONCAT(doctors.first_name, " ", doctors.last_name) as doctor_name, COUNT(*) as count')
            ->groupBy('prescriptions.doctor_id', 'doctors.first_name', 'doctors.last_name')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        return view('admin.reports.prescriptions', compact('prescriptions', 'stats', 'dailyData', 'byDoctor', 'startDate', 'endDate'));
    }

    public function doctors(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfYear()->toDateString());
        $endDate = $request->input('end_date', Carbon::now()->toDateString());

        $doctors = Doctor::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->with('user')
            ->get();

        // Statistics
        $stats = [
            'total' => Doctor::count(),
            'verified' => Doctor::where('is_verified', true)->count(),
            'pending' => Doctor::where('is_verified', false)->count(),
            'active' => Doctor::where('is_active', true)->count(),
            'registered_this_month' => Doctor::whereMonth('created_at', Carbon::now()->month)->count(),
            'registered_this_year' => Doctor::whereYear('created_at', Carbon::now()->year)->count(),
        ];

        // Registration trend
        $registrationTrend = Doctor::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Doctors by specialization
        $bySpecialization = Doctor::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('specialization, COUNT(*) as count')
            ->whereNotNull('specialization')
            ->groupBy('specialization')
            ->orderBy('count', 'desc')
            ->get();

        // Doctors by country
        $byCountry = Doctor::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('country, COUNT(*) as count')
            ->whereNotNull('country')
            ->groupBy('country')
            ->orderBy('count', 'desc')
            ->get();

        return view('admin.reports.doctors', compact('doctors', 'stats', 'registrationTrend', 'bySpecialization', 'byCountry', 'startDate', 'endDate'));
    }

    public function patients(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfYear()->toDateString());
        $endDate = $request->input('end_date', Carbon::now()->toDateString());

        $patients = Patient::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->get();

        // Statistics
        $stats = [
            'total' => Patient::count(),
            'this_month' => Patient::whereMonth('created_at', Carbon::now()->month)->count(),
            'this_year' => Patient::whereYear('created_at', Carbon::now()->year)->count(),
            'with_accounts' => Patient::whereNotNull('user_id')->count(),
        ];

        // Registration trend
        $registrationTrend = Patient::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Patients by gender
        $byGender = Patient::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('gender, COUNT(*) as count')
            ->whereNotNull('gender')
            ->groupBy('gender')
            ->get();

        // Patients by payment mode
        $byPaymentMode = Patient::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('payment_mode, COUNT(*) as count')
            ->groupBy('payment_mode')
            ->get();

        return view('admin.reports.patients', compact('patients', 'stats', 'registrationTrend', 'byGender', 'byPaymentMode', 'startDate', 'endDate'));
    }

    public function appointments(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->toDateString());
        $endDate = $request->input('end_date', Carbon::now()->toDateString());

        $appointments = Appointment::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->with(['doctor', 'patient', 'branch', 'pharmacist'])
            ->get();

        // Statistics
        $stats = [
            'total' => Appointment::count(),
            'completed' => Appointment::where('status', 'completed')->count(),
            'pending' => Appointment::where('status', 'pending')->count(),
            'cancelled' => Appointment::where('status', 'cancelled')->count(),
            'this_month' => Appointment::whereMonth('created_at', Carbon::now()->month)->count(),
            'this_year' => Appointment::whereYear('created_at', Carbon::now()->year)->count(),
        ];

        // Daily appointments
        $dailyData = Appointment::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Appointments by status
        $byStatus = Appointment::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('status, COUNT(*) as count')
            ->groupBy('status')
            ->get();

        // Appointments by branch
        $byBranch = Appointment::whereBetween('appointments.created_at', [$startDate, $endDate . ' 23:59:59'])
            ->join('branches', 'appointments.branch_id', '=', 'branches.id')
            ->selectRaw('branches.name as branch_name, COUNT(*) as count')
            ->groupBy('appointments.branch_id', 'branches.name')
            ->orderBy('count', 'desc')
            ->get();

        return view('admin.reports.appointments', compact('appointments', 'stats', 'dailyData', 'byStatus', 'byBranch', 'startDate', 'endDate'));
    }

    public function overview(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->toDateString());
        $endDate = $request->input('end_date', Carbon::now()->toDateString());

        // Overall statistics
        $stats = [
            'total_doctors' => Doctor::count(),
            'verified_doctors' => Doctor::where('is_verified', true)->count(),
            'total_patients' => Patient::count(),
            'total_prescriptions' => Prescription::count(),
            'total_appointments' => Appointment::count(),
            'completed_appointments' => Appointment::where('status', 'completed')->count(),
            'total_branches' => Branch::count(),
        ];

        // Monthly trends
        $monthlyPrescriptions = Prescription::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('MONTH(created_at) as month, YEAR(created_at) as year, COUNT(*) as count')
            ->groupBy('month', 'year')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        $monthlyAppointments = Appointment::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('MONTH(created_at) as month, YEAR(created_at) as year, COUNT(*) as count')
            ->groupBy('month', 'year')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        $monthlyPatients = Patient::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->selectRaw('MONTH(created_at) as month, YEAR(created_at) as year, COUNT(*) as count')
            ->groupBy('month', 'year')
            ->orderBy('year')
            ->orderBy('month')
            ->get();

        // Top performing doctors
        $topDoctors = Prescription::whereBetween('prescriptions.created_at', [$startDate, $endDate . ' 23:59:59'])
            ->join('doctors', 'prescriptions.doctor_id', '=', 'doctors.id')
            ->selectRaw('CONCAT(doctors.first_name, " ", doctors.last_name) as doctor_name, COUNT(*) as prescription_count')
            ->groupBy('doctor_id', 'doctors.first_name', 'doctors.last_name')
            ->orderBy('prescription_count', 'desc')
            ->limit(10)
            ->get();

        return view('admin.reports.overview', compact('stats', 'monthlyPrescriptions', 'monthlyAppointments', 'monthlyPatients', 'topDoctors', 'startDate', 'endDate'));
    }

    public function exportPrescriptions(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->toDateString());
        $endDate = $request->input('end_date', Carbon::now()->toDateString());

        $prescriptions = Prescription::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->with(['doctor', 'patient', 'appointment'])
            ->get();

        // Generate CSV
        $filename = 'prescriptions_report_' . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($prescriptions) {
            $file = fopen('php://output', 'w');
            fputcsv($file, ['Prescription ID', 'Date', 'Doctor', 'Patient', 'Diagnosis', 'Items Count']);

            foreach ($prescriptions as $prescription) {
                fputcsv($file, [
                    $prescription->prescription_number,
                    $prescription->created_at->format('Y-m-d'),
                    $prescription->doctor ? $prescription->doctor->full_name : 'N/A',
                    $prescription->patient ? $prescription->patient->full_name : 'N/A',
                    $prescription->diagnosis,
                    $prescription->items->count(),
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}

